/*************************************************/
/* Examples Program For "ET-dsPIC33WEB-V1" Board */
/* Hardware    : ET-dsPIC33WEB-V1.0              */
/* Target MCU  : dsPIC33FJ128GP708               */
/*       	   : X-TAL : 8.00 MHz                */
/* Config Fosc : XTAL = 8MHz (XT+PLL)            */
/*             : PLLPRE[4..0] = 00000 (N1=2)     */
/*             : VCO Input = 8MHz / 2 = 4MHz     */
/*             : PLLDIV[8..0] = 0x26(M=40)       */
/*             : VCO Output = 4 x 40MHz = 160MHz */
/*             : PLLPOST[1:0] = 0:0 (N2=2)       */
/*             : Fosc = 160MHz / 2 = 80MHz       */
/*             : Fcy = Fosc/2 = 80MHz /2 = 40MHz */
/* Compiler    : MPLAB + C30 V3.01  		     */
/* Write By    : Eakachai Makarn(ETT CO.,LTD.)   */
/* Last Update : 16/August/2007                  */
/* Function    : Example Use Capture 16 Bit Mode */
/*             : Used Capture Measure Pulse 1KHz */
/*************************************************/
/* Used Timer2 Count Cycle of Capture Pulse      */
/* Used Capture1(RD8) Measure Pulse 1 KHz        */
/* Minimum Period(65535 Cycle) = 1.6384mS(610Hz) */
/* Maximum Period(1000 Cycle) = 25uS(40KHz)      */
/* Display Result to Serial Port UART1(9600BPS)  */
/*************************************************/
// ET-dsPIC33WEB-V1 Hardware Board
// Fcy = 40MHz
// Tcy = 1 / 40MHz = 25nS
// Desire Timer Prescale = 1
// 1 Cycle = 25nS
// Frequency 1KHz Period = 1mS
//                       = 1mS / 25nS = 40000 Cycle
// Capture Min Frequency = 65536 Cycle
//                       = 65536 x 25nS
//                       = 1.6384mS (Approx. 610Hz)
// Capture Max Frequency = 1000 Cycle
//                       = 1000 x 25nS 
//                       = 25uS (4KHz)

#include "p33FJ128GP708.h"                             				// dsPIC33FJ128GP708 MPU Register
#include "stdio.h"													// Used "sprintf" Function

/* Setup Configuration For ET-dsPIC33WEB-V1 */
_FOSCSEL(FNOSC_PRIPLL & 											// Select Primary OSC = XT + PLL
         IESO_OFF )													// Startup With User Select OSC
_FOSC(FCKSM_CSDCMD &        										// Clock Switch Disabled	
      OSCIOFNC_OFF &												// OSC2 Pin = Clock O/P
      POSCMD_XT)													// Primary OSC = XT 
_FWDT(FWDTEN_OFF &           										// Disable Watchdog Timer 
      WINDIS_OFF &          										// Disabled Windowed WDT
      WDTPRE_PR128 &        										// Watchdog prescaler=1:128
      WDTPOST_PS32768)     											// Watchdog postscaler=1:32,768
_FPOR(FPWRT_PWR128)       											// Power-on Reset Value = 128ms
_CONFIG3(JTAGEN_OFF);												// Disable JTAG Interface
/* End of Configuration For ET-dsPIC33WEB-V1 */

unsigned int Capture_Interrupt_Count;								// Capture Edge Count
unsigned int Capture_Status;										// Capture Status
unsigned int Capture_Value1;										// 1st Capture Value Save
unsigned int Capture_Value2;		     							// 2nd Capture Value Save
char uart1_buf[40];													// "sprintf" Buffer

/* pototype  section */
void putchar1(char ch);  											// Put Char To UART-1
void print_uart1(void);												// Print String to UART1 
void delay(unsigned long int);										// Delay Time Function

/******************************/
/* Capture1 Interrupt Service */
/******************************/
void _ISR _IC1Interrupt(void)
{  
  Capture_Interrupt_Count++;										// Count Edge Detect
  if (Capture_Interrupt_Count==1) 									// First Edge Detect
  {    
	Capture_Value1 = IC1BUF; 										// Save 1st Edge Capture Value
	Capture_Status = 1;												// Set Capture Complete Status   
  }    
      
  else if (Capture_Interrupt_Count==2)								// Second Edge Detect
  { 
    Capture_Value2 = IC1BUF; 										// Save 2nd Edge Capture Value        
	Capture_Status = 1;												// Set Capture Complete Status   
    IEC0bits.IC1IE = 0;												// Disable IC1 Interrupt
  }
  
  IFS0bits.IC1IF = 0;												// Reset Capture1 Interrupt Flag  
}

int main(void)
{  
  unsigned int Period;												// Period Value

  /* Start of PLL Config Fcy = 40MIPS */
  CLKDIV &= 0xFFE0;													// PLLPRE[4..0] = 00000 (N1=2)
  CLKDIV &= 0xFF3F;													// PLLPOST[1:0] = 00 (N2=2)
  PLLFBD  = 0x26;													// PLLDIV[8..0] = 000100110(M=40)
  while(!OSCCONbits.LOCK);											// Wait PLL to Lock
  OSCCONbits.CLKLOCK = 1;											// Disable PLL Modify
  /* End of PLL Config Fly = 40MIPS */
  
  /* Start of Confog UART1 */	
  U1MODEbits.UARTEN   = 1;											// Enable RXD UART
  U1MODEbits.USIDL    = 0;											// UART Continue in IDLE
  U1MODEbits.IREN     = 0;											// IRDA Disable
  U1MODEbits.RTSMD    = 0;											// RTS = Flow Control		
  U1MODEbits.UEN1     = 0;											// Enable RX,TX Pin
  U1MODEbits.UEN0     = 0;											
  U1MODEbits.WAKE     = 0;											// Disable Wakeup
  U1MODEbits.LPBACK   = 0;											// Disable Loop Back
  U1MODEbits.ABAUD    = 0;											// Disable Auto Baudrate
  U1MODEbits.URXINV   = 0;											// RXD Pin in IDLE = "1"
  U1MODEbits.BRGH     = 0;											// Baudrate = 16 Cycle Clock
  U1MODEbits.PDSEL1   = 0;											// 8 Bit Data,No Parity
  U1MODEbits.PDSEL0   = 0;											
  U1MODEbits.STSEL    = 0;											// 1 Stop Bit
  U1STAbits.UTXISEL1  = 0;											// Interrupt By TX 1-Char
  U1STAbits.UTXISEL0  = 0;										
  U1STAbits.UTXINV	  = 0;											// IRDA Encode in IDLE = 0
  U1STAbits.UTXBRK	  = 0;											// Disable Sync Break
  U1STAbits.UTXEN     = 1;  										// Enable TXD UART 
  U1STAbits.URXISEL1  = 0;											// Interrupt BY RX 1-Char
  U1STAbits.URXISEL0  = 0;										
  U1STAbits.ADDEN	  = 0;											// Disable Address Detect

  // UART[1] Baudrate 
  // ET-dsPIC33WEB-V1 Hardware Board
  // Fcy = 40MHz
  // UxBRG = [Fcy/(16xBaud)]-1
  //       = [ 40MHz / (16 x 9600) ] - 1
  //       = 259 
  U1BRG  = 259;														// UART1 Baudrate 9600/40MHz

  // UART[1] Interrupt Control 
  IEC0bits.U1RXIE     = 0;											// Disable RX Interrupt
  IEC0bits.U1TXIE     = 0;											// Disable TX Interrupt
  /* End of Config UART1 */

  /* Start of Config Timer2 */
  T2CONbits.TON = 1;    											// ON Timer
  T2CONbits.TSIDL = 1;												// Disable Timer in IDLE Mode
  T2CONbits.T32 = 0;           										// Timer = 16 Bit Timer
  T2CONbits.TGATE = 0;												// Disable Timer Gate Control
  T2CONbits.TCKPS1 = 0;        										// Timer Prescale = 1:1			
  T2CONbits.TCKPS0 = 0;
  T2CONbits.TCS = 0;												// Timer Clock Source = Internal

  // Setup Timer Start Count
  TMR2 = 0;          												// Reset Timer to 0x0000 

  // Assign PWM Period
  PR2 = 65535;     													// Assign Period to Timer

  // Timer2 Interrupt Control 
  IEC0bits.T2IE = 0;												// Disable Timer2 Interrupt
  /* End of Config Timer2 */

  /* Start of Config IC1 */
  IC1CONbits.ICSIDL = 1;  											// Disable Capture1 in IDLE Mode             
  IC1CONbits.ICTMR = 1;			    								// Used Timer2 = Source Clock Capture1
  IC1CONbits.ICI1 = 0;            									// 1 Capture / Interrupt               
  IC1CONbits.ICI0 = 0;
  IC1CONbits.ICM2 = 0;         										// Capture1 Start on Falling Edge
  IC1CONbits.ICM1 = 1;
  IC1CONbits.ICM0 = 0;

  // IC[1] Interrupt Control 
  IEC0bits.IC1IE = 1;												// Enable IC1 Interrupt
  IPC0bits.IC1IP = 7;												// IC1 Priority = 7
  /* End of Config IC1 */
  
  // Power-On Delay
  delay(1000000);													// Power-On Delay

  // UART[1] Print String //
  sprintf(uart1_buf,"ET-dsPIC33WEB-V1.0 : Capture\n\r"); 		// Print Message String 
  print_uart1();
  sprintf(uart1_buf,"Demo IC1 Measure 1 KHz Pulse\n\r"); 		// Print Message String 
  print_uart1();
  
  /* Start Used Capture1 Measure Pulse on RD8 Pin */
  while(1)															// Loop Continue
  {      
     Capture_Interrupt_Count = 0;									// Reset Edge Count
     IEC0bits.IC1IE = 1;											// Enable IC1 Interrupt

     // First Edge Capture
     Capture_Status = 0;											// Reset Status  
     while(Capture_Status == 0);									// Wait Edge Capture Complete

     // Second Edge Capture
     Capture_Status = 0;											// Reset Status
	 while(Capture_Status == 0);									// Wait Edge Capture Complete	 
     
     if (Capture_Value2 > Capture_Value1) 							// If Not Over 65536 Cycle
     {
       Period = Capture_Value2 - Capture_Value1;
     }
     else															// If Overflow Capture
     {
       Period = (65535-Capture_Value1)+(Capture_Value2+1);       
     }
     
     // Display Capture Value in Cycle (1 / 40 MHz)
     // Frequency 1 KHz Capture Result = 40000 Cycle
     sprintf(uart1_buf,"Period = %5u Cycle\r",Period);				// Print Message String 
     print_uart1();
  } 
}	  

/****************************/
/* Write Character To UART1 */
/****************************/
void putchar1 (char c)  
{
  if (c == '\n')  													// If Line Feed(LF)
  {
    U1TXREG = 0x0D;                        							// Auto Add CR(LF+CR)
    while (!U1STAbits.TRMT);										// Wait Send Complete
    U1STAbits.TRMT = 0;												// Reset Status
  }
  U1TXREG = c;														// Send This Char
  while (!U1STAbits.TRMT);											// Wait Send Complete
  U1STAbits.TRMT = 0;												// Reset Status
}

/*************************/
/* Print String to UART1 */
/*************************/
void print_uart1(void)
{
  char *p;															// Pointer Buffer
  p = uart1_buf;													// UART1 Buffer
 
  do 																// Get char & Print Until null
  {
    putchar1(*p); 													// Write char to UART1
    p++;															// Next char
  }
  while(*p != '\0');												// End of ASCII (null)
 
  return;
}
  
/***********************/
/* Delay Time Function */
/*    1-4294967296     */
/***********************/
void delay(unsigned long int count1)
{
  while(count1 > 0) {count1--;}										// Loop Decrease Counter	
}
